<?php

namespace App\Models\HRM;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;
use OwenIt\Auditing\Contracts\Auditable;
use Illuminate\Database\Eloquent\Casts\Attribute;

class Payroll extends Model implements Auditable
{
    use HasFactory, SoftDeletes, \OwenIt\Auditing\Auditable;

    protected $fillable = [
        'uid',
        'employee_id',
        'name',
        'employeeData',
        'status',
        'date',
        'month',
        'calculation_year',
        // Attendance
        'totalWorkedRegularHours',
        'totalWorkedSuppDayHours',
        'totalWorkedSuppNightHours',
        'totalWorkedSuppHolyDaysHours',
        'totalWorkedweekendHours',
        'totalWorkedHours',
        'detailSuppl',
        'leaveDays',
        'absenceDaysUnJustified',
        'absenceDaysJustified',
        'totalAdvance',

        // SMIG
        'smig_compliant',
        'smig_minimum_required',
        'smig_deficit',

        // Compensation
        'compensation',
        'taxableCompensations',
        'nonTaxableCompensations',

        // Social security
        'cnssBase',
        'cnss',
        'cnssRate',
        'amo',
        'amoRate',
        'lossJobCompensation',
        'lossJobRate',
        'ir',
        'irRate',
        'detailIr',
        'tax_credit',
        'proFees',
        'proFeesRate',
        'retirement',
        'retirementRate',

        // Employer charges
        'employer_cnss',
        'employer_cnss_rate',
        'employer_amo',
        'employer_amo_rate',
        'employer_ipe',
        'employer_ipe_rate',
        'employer_tfp',
        'employer_tfp_rate',
        'employer_family_allowance',
        'employer_family_allowance_rate',
        'total_employer_cost',

        // Totals
        'detailGrossSalary',
        'grossSalary',
        'totalDeductions',
        'netSalary',
        'total_social_security',
        'netSalaryInWord',

        // HR
        'generatedBy',
        'approvedBy',
        'approvedAt',
        'note'
    ];

    //relation with employee
    public function employee()
    {
        return $this->belongsTo(Employee::class, 'employee_id', 'id');
    }


    protected $casts = [
        'date' => 'date',
        'approvedAt' => 'date',
        'compensation' => 'array',
        'employeeData' => 'array',
        'detailSuppl' => 'array',
        'detailGrossSalary'=> 'array',
        'detailIr'=> 'array',
        'smig_compliant' => 'boolean',

        // Decimals
        'totalWorkedRegularHours' => 'decimal:2',
        'totalWorkedSuppDayHours' => 'decimal:2',
        'totalWorkedSuppNightHours' => 'decimal:2',
        'totalWorkedSuppHolyDaysHours' => 'decimal:2',
        'totalWorkedweekendHours' => 'decimal:2',
        'totalWorkedHours' => 'decimal:2',
        'leaveDays' => 'decimal:2',
        'absenceDaysUnJustified' => 'decimal:2',
        'absenceDaysJustified' => 'decimal:2',
        'totalAdvance' => 'decimal:2',
        'cnss' => 'decimal:2',
        'amo' => 'decimal:2',
        'lossJobCompensation' => 'decimal:2',
        'ir' => 'decimal:2',
        'proFees' => 'decimal:2',
        'retirement' => 'decimal:2',
        'employer_cnss' => 'decimal:2',
        'employer_amo' => 'decimal:2',
        'employer_ipe' => 'decimal:2',
        'employer_tfp' => 'decimal:2',
        'total_employer_cost' => 'decimal:2',
        'grossSalary' => 'decimal:2',
        'totalDeductions' => 'decimal:2',
        'netSalary' => 'decimal:2',
        'total_social_security' => 'decimal:2'
    ];


    protected static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            if (empty($model->uid)) {
                $model->uid = (string) Str::ulid();
            }


            if ($model->date && empty($model->month)) {
                $model->month = $model->date->format('Y-m');
            }

            $model->calculation_year = now()->year;

            if ($model->status === 'approved') {
                $model->approvedAt = now();
                $model->approvedBy = request()->user()->name . '-' . request()->user()->id ?? 'system';
            }

        });

        static::updating(function ($model) {

            if ($model->isDirty('status') && $model->status === 'approved') {
                $model->approvedAt = now();
                $model->approvedBy = request()->user()->name . '-' . request()->user()->id ?? 'system';
            }
        });
    }
}
