<?php

namespace App\Models\HRM;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;
use OwenIt\Auditing\Contracts\Auditable;
use Illuminate\Database\Eloquent\Casts\Attribute;

class Employee extends Model implements Auditable
{
    use HasFactory, SoftDeletes, \OwenIt\Auditing\Auditable;
    protected $fillable = [
        // 🔹 Identification
        'uid',                 // Unique identifier auto
        'name',                // Full name auto
        'status',              // Employment status (active, inactive…) auto
        'matricule',           // Employee internal number/code  auto
        'pid',                 // Attendance ID auto
        'fname',               // First name
        'lname',               // Last name
        'cin',                 // Moroccan CIN
        'birthday',            // Date of birth
        'age',                 // Age
        'nationality',         // Nationality
        'passportNumber',      // Passport (for expatriates)
        'phone',               // Phone number
        'email',               // Email address
        'address',             // Physical address

        // 🔹 payrollBank
        'payrollType',
        'salaryType',          // Monthly / Hourly / Fixed
        'salary',              // Monthly salary
        'hourPrice',           // Hourly rate  //supplimentaire
        'allowances',          // JSON: primes, indemnities
        'deductions',          // JSON: deductions
        'remainingFromDeductions', //rest form minus the deduction in the payroll
        'bankAccount',         // Bank account number
        'bankName',            // Bank name
        'rib',                 // RIB
        'iban',                // IBAN
        'paymentMethod',       // Payment method (cash, transfer…)


        // 🔹 Job & Contract
        'jobTitle',            // Engineer, Technician…
        'department',          // HR, IT, Finance… setting
        'grade',               // Level or grade setting
        'employmentType',      // CDI, CDD, Stage… const
        'contractType',        // Full-time, Part-time… const
        'hireDate',            // Hiring date
        'lastTestDate',        // Last date in test period
        'contractEndDate',     // Contract end
        'noticeterminationDate',
        'noticePeriod',        // Notice period
        'terminationDate',     // Termination date (if applicable)
        'workLocation',        // Site / office
        'workingHours',        // 44h/week in morocco
        'supervisor',          // Supervisor ID or name auto
        'leaveBalance',        // Vacation days
        'sickLeaveBalance',    // Sick leave balance


        // 🔹 Social Security & Insurance
        // 🔹 Family & Social
        'maritalStatus',            // Single, Married, etc. for  payroll IR
        'childrenNumber',           // Number of children
        'relatedPerson',
        'emergencyContactName',     // Emergency contact name
        'emergencyContactPhone',    // Emergency contact phone
        'emergencyContactRelation', // Emergency contact relation
        'cnssNumber',          // CNSS
        'hasRetirementPlan',   // Retirement plan
        'hasInsurance',        // Extra insurance
        'retirementProvider',  // Retirement provider
        'insuranceCompany',    // Insurance company
        'insuranceNumber',     // Policy number
        'insuranceDays',
        'daysDeclared',        // Days Declared
        'salaryDeclared',      // Salary Declared
        'socialNotes',         // ARRAY Notes about social status

        // 🔹 Documents & Files
        'avatar',
        'cv',
        'diplome',
        'cin',
        'contract',
        'passport',

        // 🔹 HR & Performance
        'performance',         //number: Evaluation score points
        'notes',               // Internal notes
    ];

    const STATUS_ACTIVE = 'active';
    const STATUS_INACTIVE = 'inactive';

    //relation with payrolls
    public function payrolls()
    {
        return $this->hasMany(Payroll::class, 'employee_id', 'id');
    }



    // Type casting
    protected $casts = [
        'birthday'            => 'datetime',
        'hireDate' =>    'datetime',
        'contractEndDate'   => 'datetime',
        'lastTestDate'   => 'datetime',
        'noticeterminationDate'=> 'datetime',
        'terminationDate'=> 'datetime',
        'salary'              => 'decimal:2',
        'hourPrice'          => 'decimal:2',
        'daysDeclared' =>  'integer',
        'salaryDeclared' => 'decimal:2',
        'leaveBalance'  => 'decimal:2',
    ];

    protected function allowances(): Attribute
    {
        return Attribute::make(
            get: fn($value) => json_decode($value, true),
            set: fn($value) => json_encode($value),
        );
    }

    protected function deductions(): Attribute
    {
        return Attribute::make(
            get: fn($value) => json_decode($value, true),
            set: fn($value) => json_encode($value),
        );
    }

    protected function extraDeduction(): Attribute
    {
        return Attribute::make(
            get: fn($value) => json_decode($value, true),
            set: fn($value) => json_encode($value),
        );
    }

    protected function supervisor(): Attribute
    {
        return Attribute::make(
            get: fn($value) => json_decode($value, true),
            set: fn($value) => json_encode($value),
        );
    }

    protected function avatar(): Attribute
    {
        return Attribute::make(
            get: fn($value) => json_decode($value, true),
            set: fn($value) => json_encode($value),
        );
    }
    protected function cv(): Attribute
    {
        return Attribute::make(
            get: fn($value) => json_decode($value, true),
            set: fn($value) => json_encode($value),
        );
    }
    protected function diplome(): Attribute
    {
        return Attribute::make(
            get: fn($value) => json_decode($value, true),
            set: fn($value) => json_encode($value),
        );
    }

    protected function contract(): Attribute
    {
        return Attribute::make(
            get: fn($value) => json_decode($value, true),
            set: fn($value) => json_encode($value),
        );
    }
    protected function passport(): Attribute
    {
        return Attribute::make(
            get: fn($value) => json_decode($value, true),
            set: fn($value) => json_encode($value),
        );
    }


    protected static function boot()
    {
        parent::boot();
        static::creating(function ($model) {
            if (empty($model->uid)) {
                $model->uid = (string) Str::ulid();
                $model->name =  $model->lname . ' ' .  $model->fname;
                $model->status = 'active';
            }
            if (empty($model->hourPrice)) {
                $workedHours =  $model->workingHours ?? 8;
                $model->hourPrice = ($model->salaryDeclared / $model->daysDeclared) / $workedHours;
            }
        });

        static::updating(function ($model) {
            if ($model->isDirty('fname') || $model->isDirty('lname')) {
                $model->name = $model->lname . ' ' . $model->fname;
            }
            if ($model->isDirty('daysDeclared') || $model->isDirty('salaryDeclared')) {
                $workedHours =  $model->workingHours ?? 8;
                $model->hourPrice = ($model->salaryDeclared / $model->daysDeclared) / $workedHours;
            }
        });
    }
}
