<?php

namespace App\Jobs;

use Illuminate\Foundation\Queue\Queueable;
use App\Models\HRM\Employee;
use App\Models\HRM\Attendance;
use App\Models\HRM\Absence;
use App\Models\HRM\Leave;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;
use Carbon\CarbonPeriod;


class CheckAbsenceDays
{
    use Queueable;
    protected array $holydays = [
        // --- الأعياد الدينية ---
        ['name' => 'فاتح محرم', 'date' => '2025-07-07', 'days' => 1, 'type' => 'religious'], // 1 Muharram 1447
        ['name' => 'عيد المولد النبوي', 'date' => '2025-09-04', 'days' => 2, 'type' => 'religious'], // 12 Rabi' al-awwal
        ['name' => 'عيد الفطر', 'date' => '2025-03-31', 'days' => 2, 'type' => 'religious'], // 1 Shawwal
        ['name' => 'عيد الأضحى', 'date' => '2025-06-06', 'days' => 2, 'type' => 'religious'], // 10 Dhu al-Hijjah

        // --- الأعياد الوطنية ---
        ['name' => 'فاتح يناير', 'date' => '2025-01-01', 'days' => 1, 'type' => 'national'],
        ['name' => 'ذكرى تقديم عريضة الاستقلال', 'date' => '2025-01-11', 'days' => 1, 'type' => 'national'],
        ['name' => 'فاتح السنة الأمازيغية', 'date' => '2025-01-14', 'days' => 1, 'type' => 'national'],
        ['name' => 'عيد الشغل', 'date' => '2025-05-01', 'days' => 1, 'type' => 'national'],
        ['name' => 'عيد العرش', 'date' => '2025-07-30', 'days' => 1, 'type' => 'national'],
        ['name' => 'ذكرى وادي الذهب', 'date' => '2025-08-14', 'days' => 1, 'type' => 'national'],
        ['name' => 'ذكرى ثورة الملك والشعب', 'date' => '2025-08-20', 'days' => 1, 'type' => 'national'],
        ['name' => 'عيد الشباب', 'date' => '2025-08-21', 'days' => 1, 'type' => 'national'],
        ['name' => 'ذكرى المسيرة الخضراء', 'date' => '2025-11-06', 'days' => 1, 'type' => 'national'],
        ['name' => 'عيد الاستقلال', 'date' => '2025-11-18', 'days' => 1, 'type' => 'national'],
    ];

    /**
     * Create a new job instance.
     */
    public function __construct() {}

    /**
     * Execute the job.
     */
    public function handle()
    {
        try {
            $today = now()->startOfDay();
            $monthStart = now()->copy()->startOfMonth();

            Employee::where('status', Employee::STATUS_ACTIVE)
                ->chunk(50, function ($employees) use ($monthStart, $today) {

                    foreach ($employees as $employee) {

                        /* =============================
                     * 1. Attendance dates (SET)
                     * ============================= */
                        $attendanceDates = Attendance::where('employee_id', $employee->id)
                            ->whereBetween('date', [$monthStart, $today])
                            ->where('workedRegularHours', '>', 0)
                            ->pluck('date')
                            ->map(fn($d) => Carbon::parse($d)->toDateString())
                            ->flip(); // 🔥 O(1) lookup

                        /* =============================
                     * 2. Leave dates (SET)
                     * ============================= */
                        $leaveDates = [];
                        $leavePeriodLabel = null;

                        $leaves = Leave::where('employee_id', $employee->id)
                            ->whereIn('status', [
                                Leave::STATUS_ONGOING,
                                Leave::STATUS_TAKEN,
                            ])
                            ->where(function ($q) use ($monthStart, $today) {
                                $q->whereBetween('start', [$monthStart, $today])
                                    ->orWhereBetween('end', [$monthStart, $today])
                                    ->orWhere(function ($q2) use ($monthStart, $today) {
                                        $q2->where('start', '<=', $monthStart)
                                            ->where('end', '>=', $today);
                                    });
                            })
                            ->get();

                        foreach ($leaves as $leave) {
                            $period = CarbonPeriod::create($leave->start, $leave->end);
                            foreach ($period as $day) {
                                $leaveDates[$day->toDateString()] = true;
                            }
                            $leavePeriodLabel = $leave->start . ' - ' . $leave->end;
                        }

                        /* =============================
                     * 3. Holiday dates (SET)
                     * ============================= */
                        $holidayDates = [];

                        foreach ($this->holydays as $holyDay) {
                            $start = Carbon::parse($holyDay['date']);
                            $end   = $start->copy()->addDays($holyDay['days'] - 1);

                            foreach (CarbonPeriod::create($start, $end) as $day) {
                                $holidayDates[$day->toDateString()] = $holyDay;
                            }
                        }

                        /* =============================
                     * 4. Loop month days (ONCE)
                     * ============================= */
                        foreach (CarbonPeriod::create($monthStart, $today) as $day) {

                            $date = $day->toDateString();

                            // Attendance exists → remove absence
                            if (isset($attendanceDates[$date])) {
                                Absence::where('employee_id', $employee->id)
                                    ->whereDate('date', $date)
                                    ->delete();
                                continue;
                            }

                        // Holiday
                        if (array_key_exists($date, $holidayDates)) {

                            $holyDay = $holidayDates[$date];

                            Absence::firstOrCreate(
                                [
                                    'employee_id' => $employee->id,
                                    'date' => $date,
                                ],
                                [
                                    'name' => $employee->name,
                                    'status' => Absence::STATUS_JUSTIFIED,
                                    'comment' =>
                                    'Jour férié "' . ucfirst($holyDay['type']) .
                                        '" (' . $holyDay['name'] . ')',
                                ]
                            );
                            continue;
                        }


                        // Leave
                        if (isset($leaveDates[$date])) {
                                Absence::firstOrCreate(
                                    [
                                        'employee_id' => $employee->id,
                                        'date' => $date,
                                    ],
                                    [
                                        'name' => $employee->name,
                                        'status' => Absence::STATUS_JUSTIFIED,
                                        'comment' => 'En congé : ' . $leavePeriodLabel,
                                    ]
                                );
                                continue;
                            }

                            // Sunday → skip
                            if ($day->isSunday()) {
                                continue;
                            }

                            // Auto unjustified absence
                            Absence::firstOrCreate(
                                [
                                    'employee_id' => $employee->id,
                                    'date' => $date,
                                ],
                                [
                                    'name' => $employee->name,
                                    'status' => Absence::STATUS_UNJUSTIFIED,
                                    'comment' => 'Absence non justifiée genere automatique par le systeme car il n\'y a pas de pointage ce jour',
                                ]
                            );
                        }
                    }
                });
        } catch (\Throwable $e) {
            Log::error('CheckMonthlyAbsenceJob error', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
        }
    }
}
