<?php

namespace App\Http\Controllers\auth;

use App\Models\admin\User;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Exception;

class AuthController extends Controller
{
    /**
     * Register a new user.
     */
    public function register(Request $request)
    {
        try {
            $validated = $request->validate([
                'name'     => 'required|string|max:255',
                'email'    => 'required|string|email|unique:users,email',
                'password' => 'required|string|min:6',
            ]);

            $user = User::create([
                'name'     => $validated['name'],
                'email'    => $validated['email'],
                'password' => Hash::make($validated['password']),
            ]);

            $token = Auth::login($user);
            $user->last_login_at = now();
            $user->save();
            return $this->respondWithToken($token, $user, 'User registered successfully.');
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed.',
                'errors'  => $e->errors(),
            ], 422);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to register user.' . $e,
            ], 500);
        }
    }

    /**
     * Login and get a JWT token.
     */
    public function login(Request $request)
    {
        try {
            $credentials = $request->only(['email', 'password']);

            if (!$token = Auth::attempt($credentials)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized',
                ], 401);
            }


            $user = Auth::user();
            $user->last_login_at = now();
            $user->last_login_ip = $request->ip();
            $user->save();


            // fetch roles & permissions names only
            $rolesNames = $user->getRoleNames(); // ["manager","admin","owner"]
            $rolesColumns = $user->roles()
                ->get(['id', 'name', 'columns']) // ما جبتش pivot مباشرة
                ->map(function ($role) {
                    return [
                        'id'      => $role->id,
                        'name'    => $role->name,
                        'columns' => $role->columns ? json_decode($role->columns, true) : [],
                    ];
                });

            $permissions = $user->getAllPermissions()->pluck('name'); // ["user-view"]
            // prepare simplified user object
            $simplifiedUser = [
                'id'     => $user->id,
                'name'   => $user->name,
                'email'  => $user->email,
                'avatar' => $user->avatar,
                'roles'  => $rolesNames,
                'permissions'  => $permissions,
                'columns' => $rolesColumns,
            ];
            return response()->json([
                'success'      => true,
                'message'      => 'Login successful.',
                'access_token' => $token,
                'token_type'   => 'bearer',
                'expires_in'   => Auth::factory()->getTTL() * 60,
                'user'         => $simplifiedUser,
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to login.',
            ], 500);
        }
    }



    /**
     * Get the authenticated user.
     */
    public function check()
    {
        try {
            $user = Auth::user();
            // fetch roles & permissions names only
            $rolesNames = $user->getRoleNames(); // ["manager","admin","owner"]
            $rolesColumns = $user->roles()
                ->get(['id', 'name', 'columns']) // ما جبتش pivot مباشرة
                ->map(function ($role) {
                    return [
                        'id'      => $role->id,
                        'name'    => $role->name,
                        'columns' => $role->columns ? json_decode($role->columns, true) : [],
                    ];
                });
            $permissions = $user->getAllPermissions()->pluck('name'); // ["user-view"]

            // prepare simplified user object
            $simplifiedUser = [
                'id'     => $user->id,
                'name'    => $user->name,
                'email'  => $user->email,
                'avatar' => $user->avatar,
                'roles'  => $rolesNames,
                'permissions'  => $permissions,
                'columns' => $rolesColumns
            ];
            return response()->json([
                'success' => true,
                'message' => 'Authenticated user retrieved successfully.',
                'user'    =>  $simplifiedUser,
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve user info.',
            ], 500);
        }
    }

    /**
     * Logout user (invalidate token).
     */
    public function logout()
    {
        try {
            Auth::logout();
            return response()->json([
                'success' => true,
                'message' => 'Successfully logged out',
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to logout.',
            ], 500);
        }
    }

    public function changePassword(Request $request)
    {
        try {
            $user = Auth::user();
            //hach the new password
            $user->password = Hash::make($request->newPassword);
            $user->save();
            return response()->json([
                'success' => true,
                'message' => 'Password changed successfully.',
                'status' => 'success'
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to change password.',
                'status' => 'error'
            ], 500);
        }
    }

    /**
     * Build the token response.
     */
    protected function respondWithToken($token, $user = null, $message = 'Token generated.')
    {
        return response()->json([
            'success'      => true,
            'message'      => $message,
            'access_token' => $token,
            'token_type'   => 'bearer',
            'expires_in'   => Auth::factory()->getTTL() * 60,
            'user'         => $user,
        ], 200);
    }
}
