<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use App\Models\admin\User;
use Illuminate\Http\Request;
use App\Traits\CrudTrait;
use App\Models\admin\Setting;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Mail;
use App\Mail\SendUserPasswordMail;

class UserController extends Controller
{
   use CrudTrait;

    protected string $modelClass = User::class;
    protected string $dataType = 'object';

    public function index(Request $request)
    {
       return $this->getData($request);
    }

    public function store(Request $request)
    {
        try {
            $generatedPassword = null;
            $isRestored = false;

            // 🔍 Check duplicate email (including soft-deleted)
            $existingUser = User::withTrashed()
                ->where('email', $request->email)
                ->first();

            if ($existingUser) {
                if ($existingUser->trashed()) {
                    // 🔄 Restore soft-deleted user
                    $generatedPassword = $request->input('password', bin2hex(random_bytes(4)));

                    $existingUser->restore();
                    $existingUser->update([
                        'name'     => $request->name,
                        'password' => bcrypt($generatedPassword),
                    ]);

                    $isRestored = true;
                    $user = $existingUser;
                } else {
                    // ❌ Active user already exists
                    return response()->json([
                        'success' => false,
                        'message' => 'User with this email already exists',
                    ], 409);
                }
            } else {
                // 🆕 New user
                $generatedPassword = $request->input('password', bin2hex(random_bytes(4)));
                $request->merge(['password' => bcrypt($generatedPassword)]);

                $response = $this->storeData($request);
                $responseData = $response->getData(true);

                if (!($responseData['success'] ?? false)) {
                    return response()->json([
                        'success' => false,
                        'message' => $responseData['message'] ?? 'User creation failed',
                    ], 500);
                }

                $user = User::findOrFail($responseData['data']['id']);
            }

            // 🎭 Assign roles
            if ($request->filled('roles') && is_array($request->roles)) {
                $user->syncRoles($request->roles);
            }

            // 📧 Mail config
            $mailData = Setting::query()->first();
            // Configure mail settings dynamically

            config([
                'mail.default' => 'dynamic',
                'mail.mailers.dynamic' => [
                    'transport'  => 'smtp',
                    'host'       => $mailData->company['host'],
                    'port'       => (int) $mailData->company['port'],
                    'encryption' => $mailData->company['encryption'],
                    'username'   => $mailData->company['username'],
                    'password'   => $mailData->company['password'],
                ],
                'mail.from.address' => $mailData->company['fromAddress'] ?? $mailData->company['username'],
                'mail.from.name'    => $mailData->company['fromName'] ?? config('app.name'),
            ]);

            // 📤 Send email for NEW or RESTORED users
            if ($generatedPassword !== null) {
                Mail::to($user->email)
                    ->send(new SendUserPasswordMail($user->email, $generatedPassword,$user->name,$mailData->company['name']));
            }

            return response()->json([
                'success'    => true,
                'data'       => $user,
                'statusCode' => 200,
            ]);
        } catch (\Throwable $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }


    public function bulkStore(Request $request)
    {
        return $this->bulkStore($request);
    }

    public function update(Request $request)
    {
        // Find existing user
        $user = User::findOrFail($request->id);

        // Update basic fields
        $user->update(); // add other fields if needed $request->only(['name', 'email'])

        // Handle roles update
        if ($request->filled('roles')) {
            $roles = $request->roles;

            if (!is_array($roles)) {
                $roles = [$roles];
            }

            $validRoles = Role::whereIn('name', $roles)->pluck('name')->toArray();

            if (!empty($validRoles)) {
                // Replace old roles with the new ones
                $user->syncRoles($validRoles);
            }
        }

        // Reload roles for fresh response
        $user->load('roles');

        return response()->json([
            'success' => true,
            'data' => $user,
            'statusCode' => 200,
        ]);
    }

    public function bulkUpdate(Request $request)
    {
        return $this->multiUpdate($request);
    }

    public function destroy(Request $request)
    {
        return $this->destroyData($request);
    }

    public function bulkDestory(Request $request)
    {
        return $this->bulkDelete($request);
    }

    public function clone(Request $request)
    {
        return $this->cloneData($request);
    }

    public function export(Request $request)
    {
        return $this->exportData($request);
    }

    public function import(Request $request)
    {
        return $this->importData($request);
    }

    public function upload(Request $request)
    {
        return $this->uploadFile($request);
    }

    public function download(Request $request)
    {
        return $this->downloadFile($request);
    }

    public function sendUserPassword(Request $request)
    {
        $email = $request->input('email');
        $password = $request->input('password');

        if (empty($email) || empty($password)) {
            return response()->json(['error' => 'Email and password are required'], 400);
        }

        try {
            Mail::to($email)->send(new \App\Mail\SendUserPasswordMail($email, $password));
            return response()->json(['success' => 'Password email sent successfully']);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Failed to send email'], 500);
        }
    }

}
