<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use Carbon\Carbon;
use App\Models\HRM\Absence;
use App\Models\HRM\Attendance;
use App\Models\HRM\Employee;
use App\Models\HRM\Leave;
use App\Models\HRM\Payroll;
use App\Models\OPERATION\Operation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;

class StatisticController extends Controller
{
    public function hrmData()
    {
        $month = now();
        $year  = $month->year;
        $monthNumber = $month->month;
        $monthValue  = $month->format('Y-m');

        $cacheKey = "hrm_dashboard_{$monthValue}";

        return Cache::remember($cacheKey, 300, function () use (
            $month,
            $year,
            $monthNumber,
            $monthValue
        ) {

            // --------------------------------------------------
            // 1️⃣ Departments (once)
            // --------------------------------------------------
            $departments = Employee::query()
                ->whereNotNull('department')
                ->distinct()
                ->orderBy('department')
                ->pluck('department')
                ->toArray();

            // --------------------------------------------------
            // 2️⃣ Employees (minimal select)
            // --------------------------------------------------
            $employees = Employee::query()
                ->select('id', 'department', 'status')
                ->get();

            $activeEmployees = $employees->where('status', Employee::STATUS_ACTIVE);

            $totalEmployees = $activeEmployees->count();

            // --------------------------------------------------
            // 3️⃣ Absences / Leaves / Attendances (month scoped)
            // --------------------------------------------------
            $absences = Absence::query()
                ->select('employee_id', 'date')
                ->whereMonth('date', $monthNumber)
                ->whereYear('date', $year)
                ->get();

            $leaves = Leave::query()
                ->select('employee_id', 'start', 'end')
                ->where(function ($q) use ($month, $year) {
                    $q->whereMonth('start', $month->month)
                        ->whereYear('start', $year)
                        ->orWhereMonth('end', $month->month)
                        ->whereYear('end', $year);
                })
                ->get();

            $attendances = Attendance::query()
                ->select('employee_id', 'date')
                ->whereMonth('date', $monthNumber)
                ->whereYear('date', $year)
                ->get();

            // --------------------------------------------------
            // 4️⃣ Tiles counts (distinct employees)
            // --------------------------------------------------
            $absenceCount    = $absences->pluck('employee_id')->unique()->count();
            $leaveCount      = $leaves->pluck('employee_id')->unique()->count();
            $attendanceCount = $attendances->pluck('employee_id')->unique()->count();

            // --------------------------------------------------
            // 5️⃣ Department mapping helpers
            // --------------------------------------------------
            $employeesByDept = $activeEmployees
                ->groupBy('department')
                ->map->count();

            $absenceByDept = $absences
                ->groupBy(function ($a) use ($employees) {
                    return optional(
                        $employees->firstWhere('id', $a->employee_id)
                    )->department;
                })
                ->map(fn($items) => $items->pluck('employee_id')->unique()->count());

            $leaveByDept = $leaves
                ->groupBy(function ($l) use ($employees) {
                    return optional(
                        $employees->firstWhere('id', $l->employee_id)
                    )->department;
                })
                ->map(fn($items) => $items->pluck('employee_id')->unique()->count());

            $attendanceByDept = $attendances
                ->groupBy(function ($a) use ($employees) {
                    return optional(
                        $employees->firstWhere('id', $a->employee_id)
                    )->department;
                })
                ->map(fn($items) => $items->pluck('employee_id')->unique()->count());

            $map = fn($data) => array_map(
                fn($dept) => (int) ($data[$dept] ?? 0),
                $departments
            );

            // --------------------------------------------------
            // 6️⃣ Payroll (single query)
            // --------------------------------------------------
            $payrolls = Payroll::query()
                ->select(
                    'netSalary',
                    'total_employer_cost',
                    'grossSalary',
                    'leaveDays',
                    'absenceDaysJustified'
                )
                ->where('calculation_year', $year)
                ->where('month', $monthValue)
                ->whereNull('deleted_at')
                ->get();

            $masseSalariale     = $payrolls->sum('netSalary');
            $chargesPatronales = $payrolls->sum('total_employer_cost');

            $congesFinancier = $payrolls->sum(
                fn($p) => ($p->grossSalary / 26) * $p->leaveDays
            );

            $absencesJustifieesFinancier = $payrolls->sum(
                fn($p) => ($p->grossSalary / 26) * $p->absenceDaysJustified
            );

            // --------------------------------------------------
            // 7️⃣ Final response (UNCHANGED)
            // --------------------------------------------------
            return [
                'tiles' => [
                    'employee'   => $totalEmployees,
                    'absence'    => $absenceCount,
                    'leave'      => $leaveCount,
                    'attendance' => $attendanceCount,
                ],
                'dataSource' => [
                    ['type', ...$departments],
                    ['employee', ...$map($employeesByDept)],
                    ['absence', ...$map($absenceByDept)],
                    ['conge', ...$map($leaveByDept)],
                    ['attendance', ...$map($attendanceByDept)],
                ],
                'data' => [
                    ['value' => round($masseSalariale, 2), 'name' => 'Masse salariale'],
                    ['value' => round($chargesPatronales, 2), 'name' => 'Charges patronales'],
                    ['value' => round($congesFinancier, 2), 'name' => 'Congés (coût)'],
                    ['value' => round($absencesJustifieesFinancier, 2), 'name' => 'Absences justifiées (coût)'],
                ],
            ];
        });
    }



    public function saleData(Request $request)
    {
        $from = Carbon::parse($request->from)->startOfDay();
        $to   = Carbon::parse($request->to)->endOfDay();

        $cacheKey = "sales:kpis:{$from->format('Ymd')}:{$to->format('Ymd')}";

        return Cache::remember($cacheKey, now()->addMinutes(5), function () use ($from, $to) {

            $docTypes = [
                'invoice'  => 'invoicesale',
                'estimate' => 'estimate',
                'delivery' => 'deliverynote',
            ];

            /** --------------------------------
             * Load operations once
             * -------------------------------- */
            $operations = Operation::query()
                ->select('id', 'date', 'docType', 'status', 'totalTtc', 'realCostTotalHt')
                ->whereBetween('date', [$from, $to])
                ->whereIn('docType', $docTypes)
                ->get()
                ->groupBy('docType');

            $invoices  = $operations->get($docTypes['invoice'], collect());
            $estimates = $operations->get($docTypes['estimate'], collect());
            $deliveries = $operations->get($docTypes['delivery'], collect());

            /** --------------------------------
             * Core KPIs
             * -------------------------------- */
            $turnover = $invoices->sum('totalTtc');

            $paidInvoices = $invoices->where('status', 'paid');
            $cashIn       = $paidInvoices->sum('totalTtc');

            $totalInvoicesCount = $invoices->count();
            $paidInvoicesCount  = $paidInvoices->count();

            $invoicePaidRate = $totalInvoicesCount > 0
                ? round(($paidInvoicesCount / $totalInvoicesCount) * 100, 2)
                : 0;

            /** --------------------------------
             * Outstanding / Risk
             * -------------------------------- */
            $unpaidInvoices = $invoices->where('status', '!=', 'paid');

            $outstandingAmount = $unpaidInvoices->sum('totalTtc');
            $outstandingCount  = $unpaidInvoices->count();

            /** --------------------------------
             * Profitability
             * -------------------------------- */
            $grossProfit = $invoices->sum(
                fn($inv) =>
                $inv->totalTtc - ($inv->realCostTotalHt ?? 0)
            );

            /** --------------------------------
             * Conversion rates
             * -------------------------------- */
            $estimateToInvoiceRate = $estimates->count() > 0
                ? round(($totalInvoicesCount / $estimates->count()) * 100, 2)
                : 0;

            /** --------------------------------
             * Previous period comparison
             * -------------------------------- */
            $previousTurnover = Operation::query()
                ->whereBetween('date', [
                    (clone $from)->subMonth(),
                    (clone $to)->subMonth(),
                ])
                ->where('docType', $docTypes['invoice'])
                ->sum('totalTtc');

            $growthRate = $previousTurnover > 0
                ? round((($turnover - $previousTurnover) / $previousTurnover) * 100, 2)
                : 0;

            /** --------------------------------
             * Charts
             * -------------------------------- */
            $paymentStatusChart = $invoices
                ->groupBy('status')
                ->map(fn($items, $status) => [
                    'label' => ucfirst($status),
                    'value' => $items->sum('totalTtc'),
                ])
                ->values();

            /** --------------------------------
             * API Response
             * -------------------------------- */
            return response()->json([
                'period' => [
                    'from' => $from->toDateString(),
                    'to'   => $to->toDateString(),
                ],

                'kpis' => [
                    'turnover' => [
                        'value' => $turnover,
                        'growthRate' => $growthRate,
                    ],
                    'cashIn' => $cashIn,
                    'grossProfit' => $grossProfit,
                    'outstanding' => [
                        'amount' => $outstandingAmount,
                        'count'  => $outstandingCount,
                    ],
                ],

                'rates' => [
                    'invoicePaidRate'      => $invoicePaidRate,
                    'estimateToInvoiceRate' => $estimateToInvoiceRate,
                ],

                'charts' => [
                    'paymentStatus' => $paymentStatusChart,
                ],

                'lists' => [
                    'paidInvoices' => $paidInvoices->values(),
                    'unpaidInvoices' => $unpaidInvoices->values(),
                    'estimates' => $estimates->values(),
                    'deliveries' => $deliveries->values(),
                ],
            ]);
        });
    }
}
